import Graph from "./graph";
import Point from "./point";

var graph = new Graph();
var selectedEdges = new Array<[Point,Point]>();
var selectColor = "#FFD300";
var vertexSize = 10;
var vertexColor = "blue";
var edgeColor = "red";
var selectedVertex: Point | undefined;
var backgroundImage = document.createElement('img'); 
var x_guide = 0;
var y_guide = 0;

const c: HTMLCanvasElement = <any>document.getElementById("myCanvas");
const ctx = c.getContext("2d");

c.addEventListener('mousemove', redraw);
c.addEventListener('click', clickCanvas);

const graphLoader: HTMLInputElement = <any>document.getElementById("graphLoader").addEventListener("change",
function (){
  selectedEdges.length = 0;
  selectedVertex = undefined;

  let file = (<any>document.querySelector("#graphLoader")).files[0];
  var reader = new FileReader();
  
  reader.addEventListener('load', function () {
    let graphString = <String>reader.result;
    let vertices = graphString.split("\n#")[0];
    let edges = graphString.split("\n#")[1];
    let formatMap: Map<number,Point> = new Map();
    graph = new Graph();
    for(let vertex of vertices.split("\n")){
      let id = vertex.split(":")[0];
      let pos = vertex.split(":")[1];
      let x = pos.split(",")[0];
      let y = pos.split(",")[1];
      let point = new Point(Number(x), Number(y));
      formatMap.set(Number(id), point);
      graph.addVertex(point);
    }

    for(let edge of edges.slice(1,-1).split("\n")){
      graph.addEdge(formatMap.get(Number(edge.split(" ")[0])), formatMap.get(Number(edge.split(" ")[1])));  
    }

  });
  
  if(file){
      reader.readAsBinaryString(file);
  }
});

const clearGraph: HTMLButtonElement = <any>document.getElementById("clearGraph").addEventListener("click", 
function(event){
  selectedEdges.length = 0;
  selectedVertex = undefined;
  graph = new Graph();
  redraw(event);
});

const downloadGraph : HTMLButtonElement = <any>document.getElementById("download").addEventListener("click",
function(){
  const element = document.createElement('a');
  element.setAttribute('href', 'data:text/plain;charset=utf-8,' + encodeURIComponent(graph.toString()));
  element.setAttribute('download', "graph.txt");

  element.style.display = 'none';
  document.body.appendChild(element);

  element.click();  
  document.body.removeChild(element);
});

const deleteBackground : HTMLButtonElement = <any>document.getElementById("deleteBackground").addEventListener("click",
function(event){
  backgroundImage.src = "";
  redraw(event);
});

const createDelaunay: HTMLButtonElement = <any>document.getElementById("createDelaunay").addEventListener("click",
function(event){
  selectedVertex = undefined;
  selectedEdges.length = 0;
  graph = graph.createDelaunayTriangulation();
  redraw(event);
});

const compareDelaunay: HTMLButtonElement = <any>document.getElementById("compareDelaunay").addEventListener("click",
function(event){
  selectedVertex = null;
  let graphDelaunay = graph.createDelaunayTriangulation();

  for(let [vertex, adjVertices] of graph.adjList){
    for(let adjVertex of adjVertices){
      if(!(graphDelaunay.adjList.get(vertex).indexOf(adjVertex) > -1) && 
      !(graphDelaunay.adjList.get(adjVertex).indexOf(vertex) > -1)){
        selectedEdges.push([vertex, adjVertex]);
      }
    }
  }
  redraw(event);
});

const compareDelaunayDownload: HTMLButtonElement = <any>document.getElementById("compareDelaunayDownload").addEventListener("click",
function(){
  selectedVertex = undefined;
    let graphDelaunay = graph.createDelaunayTriangulation();
    let message = "Graph: " + "\n";
    message += graph.toString();
    message += "Delaunay Triangulation: " + "\n";
    message += graphDelaunay.toString();
  
    let noEGDT = 0;
    let noEG = 0;
    let noDT = 0;

    for(let [, adjVertices] of graph.adjList){
      noEG += adjVertices.length;
    }

    for(let [, adjVertices] of graphDelaunay.adjList){
      noDT += adjVertices.length;
    }
  
    for(let [vertex, adjVertices] of graph.adjList){
      for(let adjVertex of adjVertices){
        if(graphDelaunay.adjList.get(vertex).indexOf(adjVertex) > -1 || 
        graphDelaunay.adjList.get(adjVertex).indexOf(vertex) > -1){
          noEGDT+=1;
        }
      }
    }

  message += "Number of edges in Graph:\n";
  message += noEG + "\n";
  message += "Number of edges in DT:\n";
  message += noDT + "\n";
  message += "Number of Graph edges in DT:\n";
  message += noEGDT + "\n";
  message += "Percentage of Graph edges in DT:\n";
  message += noEGDT/noEG*100;

  var element = document.createElement('a');
  element.setAttribute('href', 'data:text/plain;charset=utf-8,' + encodeURIComponent(message));
  element.setAttribute('download', "graphDelaunay.txt");

  element.style.display = 'none';
  document.body.appendChild(element);

  element.click();

  document.body.removeChild(element);
});

let imgLoader : HTMLInputElement = <any>document.getElementById("imgLoader").addEventListener("change",
function(){
  let file = (<any>document.querySelector('#imgLoader')).files[0];
  var reader = new FileReader();
 
  reader.addEventListener("load", function (event) {
    backgroundImage.src = <string> reader.result;
    
    setTimeout(() => {
      let ratioW =  c.width /backgroundImage.width;
      let ratioH = c.height / backgroundImage.height;

      if(ratioH > ratioW){
         backgroundImage.height = backgroundImage.height * ratioW;
         backgroundImage.width = c.width;
      }else{   
        backgroundImage.width = backgroundImage.width * ratioH;
        backgroundImage.height = c.height;
      }     
      ctx.drawImage(backgroundImage, c.width / 2 - backgroundImage.width / 2, c.height / 2 - backgroundImage.height / 2, backgroundImage.width, backgroundImage.height);
    }, 0);
  },false);
    
  if (file) {
    reader.readAsDataURL(file);
  }
});

const setGrid: HTMLButtonElement = <any>document.getElementById("setGrid").addEventListener("click",
function(event){
  let x_input: HTMLInputElement = <any>document.getElementById("x_guide");
  let y_input: HTMLInputElement = <any>document.getElementById("y_guide");
  x_guide = Number(x_input.value);
  y_guide = Number(y_input.value);
  redraw(event);
});

function getMousePos(event: MouseEvent) {
  let rect = c.getBoundingClientRect();
  let mousePos = new Point(Math.round(event.clientX - rect.left), Math.round(event.clientY - rect.top));
  return mousePos;
}

function findGuideLinePosition(mousePos: Point) {
  if (x_guide > 0) {
    let resolution_x = c.width / (x_guide + 1);
    mousePos.x = mousePos.x / resolution_x;
    mousePos.x = Math.round(mousePos.x);
    mousePos.x = mousePos.x * resolution_x;
  }
  if (y_guide > 0) {
    let resolution_y = c.width / (y_guide + 1);
    mousePos.y = mousePos.y / resolution_y;
    mousePos.y = Math.round(mousePos.y);
    mousePos.y = mousePos.y * resolution_y;
  }
  return mousePos;
}

function checkInside(v1: Point, v2: Point) {
  if (v1.x > v2.x - vertexSize &&
    v1.x < v2.x + vertexSize &&
    v1.y > v2.y - vertexSize &&
    v1.y < v2.y + vertexSize) {
    return true;
  }
  return false;
}

function redraw(event: MouseEvent) {
  ctx.beginPath();
  ctx.clearRect(0, 0, c.width, c.height);
  ctx.fillStyle = "#ff9472";
  ctx.lineWidth = 1;
  let mousePos = getMousePos(event);
  const message = "x: " + mousePos.x + " y: " + mousePos.y + " selected: " + selectedVertex;
  ctx.strokeStyle = "#ff9472";
  if (backgroundImage != undefined) {
    ctx.drawImage(backgroundImage, c.width / 2 - backgroundImage.width / 2, c.height / 2 - backgroundImage.height / 2, backgroundImage.width, backgroundImage.height);
  }
  for (let i = 0; i < x_guide; i++) {
    ctx.beginPath();
    ctx.moveTo(c.width / (x_guide + 1) * (i + 1), 0);
    ctx.lineTo(c.width / (x_guide + 1) * (i + 1), c.height);
    ctx.stroke();
  }
  for (let i = 0; i < y_guide; i++) {
    ctx.beginPath();
    ctx.moveTo(0, c.height / (y_guide + 1) * (i + 1));
    ctx.lineTo(c.width, c.height / (y_guide + 1) * (i + 1));
    ctx.stroke();
  }
  ctx.lineWidth = 3;
  for (const [vertex, ] of graph.adjList) {
    for (const adjVertex of graph.adjList.get(vertex)) {
      ctx.beginPath();
      ctx.strokeStyle = edgeColor;
      ctx.moveTo(vertex.x, vertex.y);
      ctx.lineTo(adjVertex.x, adjVertex.y);
      ctx.stroke();
    }
  }

  ctx.strokeStyle = selectColor;
  for (let [x,y] of selectedEdges) {
    ctx.beginPath();
    ctx.moveTo(x.x, x.y);
    ctx.lineTo(y.x, y.y);
    ctx.stroke();
  }

  for (const vertex of graph.adjList.keys()) {
    ctx.fillStyle = vertexColor;
    ctx.fillRect(vertex.x - vertexSize / 2, vertex.y - vertexSize / 2, vertexSize, vertexSize);
  }

  ctx.fillStyle = selectColor;
  if (selectedVertex != null) {
    ctx.fillRect(selectedVertex.x - vertexSize / 2, selectedVertex.y - vertexSize / 2, vertexSize, vertexSize);
  }
}

function clickVertex(event: MouseEvent) {
  let mousePos = getMousePos(event);
  let insideOfVertex = false;
  let nextVertex;
  mousePos = findGuideLinePosition(mousePos);
  for (let vertex of graph.adjList.keys()) {
    if (checkInside(vertex, mousePos)) {
      insideOfVertex = true;
      nextVertex = vertex;
    }
  }
  if (insideOfVertex && nextVertex == selectedVertex) {
    selectedVertex = null;
  }
  else if (insideOfVertex) {
    selectedVertex = nextVertex;
  }
  else {
    graph.addVertex(mousePos);
  }
}
function clickEdge(event: MouseEvent) {
  if (selectedVertex == null) {
    clickVertex(event);
  }
  else {
    const mousePos = getMousePos(event);
    for (let vertex of graph.adjList.keys()) {
      if (checkInside(vertex, mousePos)) {
        graph.addEdge(vertex, selectedVertex);
      }
    }
  }
}

function removeVertex(event: MouseEvent) {
  const mousePos = getMousePos(event);
  let insideOfVertex = false;
  let nextVertex;
  for (let vertex of graph.adjList.keys()) {
    if (checkInside(vertex, mousePos)) {
      insideOfVertex = true;
      nextVertex = vertex;
    }
  }
  if (insideOfVertex) {
    graph.removeVertex(nextVertex);
  }
  if (nextVertex == selectedVertex) {
    selectedVertex = null;
  }
}

function removeEdge(event: MouseEvent) {
  if (selectedVertex == null) {
    clickVertex(event);
  }
  else {
    const mousePos = getMousePos(event);
    for (let point of graph.adjList.keys()) {
      if (checkInside(point, mousePos)) {
        graph.removeEdge(point, selectedVertex);
      }
    }
  }
}

function clickCanvas(event: MouseEvent) {
  selectedEdges.length = 0;
  if (event.shiftKey && event.ctrlKey) {
    removeEdge(event);
  }
  else if (event.ctrlKey) {
    removeVertex(event);
  }
  else if (event.shiftKey) {
    clickEdge(event);
  }
  else {
    clickVertex(event);
  }
  redraw(event);
}

